import { useState, useRef, useEffect } from 'react'
import { useNavigate } from 'react-router-dom'
import { Megaphone, Mail, BarChart3, Target, Calendar, CheckCircle, Shield, Clock, Award, Users, HelpCircle, MessageSquare, Quote, Star } from 'lucide-react'
import ImageSlider from '../components/ImageSlider'
import './Enroll.css'

const Enroll = () => {
  const navigate = useNavigate()
  const [selectedDirection, setSelectedDirection] = useState('')
  const [startDate, setStartDate] = useState('')
  const [isSubmitted, setIsSubmitted] = useState(false)

  const directions = [
    { 
      id: 'smm', 
      name: 'SMM', 
      icon: Megaphone,
      color: '#EC4899',
      description: 'Социальные сети и контент-маркетинг. Научитесь создавать вирусный контент и управлять сообществами.',
      duration: '8 недель',
      students: '1200+',
      rating: '4.9'
    },
    { 
      id: 'email', 
      name: 'Email-маркетинг', 
      icon: Mail,
      color: '#1E40AF',
      description: 'Эффективные email-рассылки и автоматизация. Повышайте конверсию с помощью персонализированных писем.',
      duration: '6 недель',
      students: '800+',
      rating: '4.8'
    },
    { 
      id: 'analytics', 
      name: 'Аналитика', 
      icon: BarChart3,
      color: '#06B6D4',
      description: 'Работа с данными и метриками. Научитесь анализировать эффективность маркетинговых кампаний.',
      duration: '10 недель',
      students: '950+',
      rating: '4.9'
    },
    { 
      id: 'context', 
      name: 'Контекстная реклама', 
      icon: Target,
      color: '#7C3AED',
      description: 'Яндекс.Директ и Google Ads. Создавайте эффективные рекламные кампании и оптимизируйте бюджет.',
      duration: '8 недель',
      students: '1100+',
      rating: '4.8'
    }
  ]

  const handleSubmit = (e) => {
    e.preventDefault()
    
    if (!selectedDirection || !startDate) {
      alert('Пожалуйста, выберите направление и дату старта')
      return
    }

    const saved = localStorage.getItem('marketingCourses')
    const courses = saved ? JSON.parse(saved) : []
    
    const directionData = directions.find(d => d.id === selectedDirection)
    
    const newCourse = {
      id: Date.now(),
      module: directionData.name,
      mentor: 'Преподаватель будет назначен',
      startDate: startDate,
      status: 'active',
      certificate: null,
      grade: null
    }
    
    courses.push(newCourse)
    localStorage.setItem('marketingCourses', JSON.stringify(courses))
    
    setIsSubmitted(true)
    
    setTimeout(() => {
      navigate('/')
    }, 2000)
  }

  const today = new Date()
  const maxDate = new Date(today.getTime() + 90 * 24 * 60 * 60 * 1000)
  const minDateStr = today.toISOString().split('T')[0]
  const maxDateStr = maxDate.toISOString().split('T')[0]

  return (
    <div className="enroll-page-marketing">
      <div className="container">
        <section className="enroll-header-marketing">
          <div className="header-badge-marketing">
            <Target size={20} />
            <span>ЗАПИСЬ НА КУРС</span>
          </div>
          <h1 className="enroll-title-marketing">
            Записаться на <span className="title-accent-marketing">курс</span>
          </h1>
          <p className="enroll-description-marketing">
            Выберите направление и дату старта. Все курсы включают практические задания и обратную связь от наставников.
          </p>
        </section>

        <section className="enroll-form-section-marketing">
          <form className="form-marketing" onSubmit={handleSubmit}>
            <div className="directions-selection-marketing">
              {directions.map((direction) => {
                const Icon = direction.icon
                return (
                  <button
                    key={direction.id}
                    type="button"
                    className={`direction-card-marketing ${selectedDirection === direction.id ? 'selected' : ''}`}
                    onClick={() => setSelectedDirection(direction.id)}
                    style={{ '--direction-color': direction.color }}
                  >
                    <div className="direction-icon-wrapper-marketing">
                      <Icon size={40} />
                    </div>
                    <h3 className="direction-name-marketing">{direction.name}</h3>
                    <p className="direction-desc-marketing">{direction.description}</p>
                    <div className="direction-stats-marketing">
                      <div className="direction-stat-marketing">
                        <span className="stat-label-marketing">Длительность:</span>
                        <span className="stat-value-marketing">{direction.duration}</span>
                      </div>
                      <div className="direction-stat-marketing">
                        <span className="stat-label-marketing">Студентов:</span>
                        <span className="stat-value-marketing">{direction.students}</span>
                      </div>
                      <div className="direction-stat-marketing">
                        <span className="stat-label-marketing">Рейтинг:</span>
                        <span className="stat-value-marketing">{direction.rating}</span>
                      </div>
                    </div>
                    {selectedDirection === direction.id && (
                      <div className="check-badge-marketing">
                        <CheckCircle size={24} />
                      </div>
                    )}
                  </button>
                )
              })}
            </div>

            {selectedDirection && (
              <div className="date-section-marketing">
                <div className="date-label-marketing">
                  <Calendar size={20} />
                  <span>Дата старта курса</span>
                </div>
                <input
                  type="date"
                  className="date-input-marketing"
                  value={startDate}
                  onChange={(e) => setStartDate(e.target.value)}
                  min={minDateStr}
                  max={maxDateStr}
                  required
                />
              </div>
            )}

            <button 
              type="submit" 
              className="submit-btn-marketing" 
              disabled={!selectedDirection || !startDate || isSubmitted}
            >
              <span>Записаться на курс</span>
            </button>

            {isSubmitted && (
              <div className="success-message-marketing">
                <CheckCircle size={32} />
                <span>Запись успешно оформлена!</span>
              </div>
            )}
          </form>
        </section>

        <ImageSlider />

        <BenefitsSection />

        <TestimonialsSection />

        <FAQSection />
      </div>
    </div>
  )
}

const BenefitsSection = () => {
  const sectionRef = useRef(null)
  const [isVisible, setIsVisible] = useState(false)

  useEffect(() => {
    const observer = new IntersectionObserver(
      ([entry]) => {
        if (entry.isIntersecting) {
          setIsVisible(true)
        }
      },
      { threshold: 0.1 }
    )

    if (sectionRef.current) {
      observer.observe(sectionRef.current)
    }

    return () => {
      if (sectionRef.current) {
        observer.unobserve(sectionRef.current)
      }
    }
  }, [])

  const benefits = [
    {
      icon: Shield,
      title: 'Практический опыт',
      description: 'Реальные кейсы и проекты от ведущих маркетологов',
      color: 'var(--marketing-blue-light)'
    },
    {
      icon: Clock,
      title: 'Гибкий график',
      description: 'Обучение в удобное время с доступом к материалам 24/7',
      color: 'var(--marketing-purple-light)'
    },
    {
      icon: Award,
      title: 'Сертификат',
      description: 'Официальный сертификат по завершении курса',
      color: 'var(--marketing-cyan)'
    },
    {
      icon: Users,
      title: 'Сообщество',
      description: 'Общение с единомышленниками и обмен опытом',
      color: 'var(--marketing-pink)'
    }
  ]

  return (
    <section 
      ref={sectionRef}
      className={`benefits-section-marketing ${isVisible ? 'visible' : ''}`}
    >
      <div className="container">
        <div className="benefits-header-marketing">
          <h2 className="benefits-title-marketing">Преимущества обучения</h2>
          <p className="benefits-description-marketing">
            Почему выбирают наши курсы по маркетингу
          </p>
        </div>
        <div className="benefits-grid-marketing">
          {benefits.map((benefit, index) => {
            const Icon = benefit.icon
            return (
              <div 
                key={index} 
                className="benefit-card-marketing"
                style={{ 
                  animationDelay: `${index * 0.1}s`,
                  '--benefit-color': benefit.color
                }}
              >
                <div className="benefit-icon-wrapper-marketing">
                  <Icon size={32} />
                </div>
                <h3 className="benefit-title-marketing">{benefit.title}</h3>
                <p className="benefit-text-marketing">{benefit.description}</p>
              </div>
            )
          })}
        </div>
      </div>
    </section>
  )
}

const TestimonialsSection = () => {
  const sectionRef = useRef(null)
  const [isVisible, setIsVisible] = useState(false)

  useEffect(() => {
    const observer = new IntersectionObserver(
      ([entry]) => {
        if (entry.isIntersecting) {
          setIsVisible(true)
        }
      },
      { threshold: 0.1 }
    )

    if (sectionRef.current) {
      observer.observe(sectionRef.current)
    }

    return () => {
      if (sectionRef.current) {
        observer.unobserve(sectionRef.current)
      }
    }
  }, [])

  const testimonials = [
    {
      name: 'Елена Смирнова',
      role: 'Маркетолог',
      text: 'Курс по SMM полностью изменил мой подход к работе. Теперь я могу создавать контент, который действительно работает.',
      rating: 5,
      avatar: 'ЕС'
    },
    {
      name: 'Александр Козлов',
      role: 'Директор по маркетингу',
      text: 'Отличный курс по аналитике! Научился работать с данными и принимать решения на основе метрик.',
      rating: 5,
      avatar: 'АК'
    },
    {
      name: 'Ольга Новикова',
      role: 'Email-маркетолог',
      text: 'Практические задания и обратная связь от наставника помогли мне значительно улучшить результаты рассылок.',
      rating: 5,
      avatar: 'ОН'
    }
  ]

  return (
    <section 
      ref={sectionRef}
      className={`testimonials-section-marketing ${isVisible ? 'visible' : ''}`}
    >
      <div className="container">
        <div className="testimonials-header-marketing">
          <h2 className="testimonials-title-marketing">Отзывы студентов</h2>
          <p className="testimonials-description-marketing">
            Что говорят наши выпускники о курсах
          </p>
        </div>
        <div className="testimonials-grid-marketing">
          {testimonials.map((testimonial, index) => (
            <div 
              key={index} 
              className="testimonial-card-marketing"
              style={{ animationDelay: `${index * 0.1}s` }}
            >
              <div className="testimonial-quote-marketing">
                <Quote size={32} />
              </div>
              <div className="testimonial-stars-marketing">
                {Array.from({ length: 5 }).map((_, i) => (
                  <Star
                    key={i}
                    size={18}
                    fill="#F59E0B"
                    color="#F59E0B"
                  />
                ))}
              </div>
              <p className="testimonial-text-marketing">{testimonial.text}</p>
              <div className="testimonial-author-marketing">
                <div className="testimonial-avatar-marketing">
                  {testimonial.avatar}
                </div>
                <div className="testimonial-info-marketing">
                  <div className="testimonial-name-marketing">{testimonial.name}</div>
                  <div className="testimonial-role-marketing">{testimonial.role}</div>
                </div>
              </div>
            </div>
          ))}
        </div>
      </div>
    </section>
  )
}

const FAQSection = () => {
  const sectionRef = useRef(null)
  const [isVisible, setIsVisible] = useState(false)
  const [openIndex, setOpenIndex] = useState(null)

  useEffect(() => {
    const observer = new IntersectionObserver(
      ([entry]) => {
        if (entry.isIntersecting) {
          setIsVisible(true)
        }
      },
      { threshold: 0.1 }
    )

    if (sectionRef.current) {
      observer.observe(sectionRef.current)
    }

    return () => {
      if (sectionRef.current) {
        observer.unobserve(sectionRef.current)
      }
    }
  }, [])

  const faqs = [
    {
      question: 'Как проходит обучение?',
      answer: 'Обучение проходит онлайн в формате видео-лекций, практических заданий и вебинаров с наставниками. Все материалы доступны 24/7.'
    },
    {
      question: 'Нужен ли опыт в маркетинге?',
      answer: 'Нет, наши курсы подходят как для начинающих, так и для опытных специалистов. Каждый курс имеет свой уровень сложности.'
    },
    {
      question: 'Когда я получу сертификат?',
      answer: 'Сертификат выдается после успешного завершения всех модулей курса и выполнения финального проекта.'
    },
    {
      question: 'Можно ли вернуть деньги?',
      answer: 'Да, мы предоставляем гарантию возврата средств в течение 14 дней после начала обучения, если курс вам не подошел.'
    }
  ]

  return (
    <section 
      ref={sectionRef}
      className={`faq-section-marketing ${isVisible ? 'visible' : ''}`}
    >
      <div className="container">
        <div className="faq-header-marketing">
          <h2 className="faq-title-marketing">Часто задаваемые вопросы</h2>
          <p className="faq-description-marketing">
            Ответы на популярные вопросы о наших курсах
          </p>
        </div>
        <div className="faq-list-marketing">
          {faqs.map((faq, index) => (
            <div 
              key={index} 
              className={`faq-item-marketing ${openIndex === index ? 'open' : ''}`}
              style={{ animationDelay: `${index * 0.1}s` }}
            >
              <button
                className="faq-question-marketing"
                onClick={() => setOpenIndex(openIndex === index ? null : index)}
              >
                <HelpCircle size={24} />
                <span>{faq.question}</span>
                <div className="faq-icon-marketing">
                  <span className={`faq-arrow-marketing ${openIndex === index ? 'open' : ''}`}>▼</span>
                </div>
              </button>
              {openIndex === index && (
                <div className="faq-answer-marketing">
                  <p>{faq.answer}</p>
                </div>
              )}
            </div>
          ))}
        </div>
      </div>
    </section>
  )
}

export default Enroll

